#include "dm_os_linux.h"
#include "dm_linux_internal.h"

/**
 * DM-style implementation of extra functions that are not available in the
 * core DM interfaces.
 */

void HwDelayUsec(UINT32 us)
{
    udelay(us);
}

INT32 OsQueueBytes(UINT32 handle, const void* item, UINT32 itemSize)
{
    INT32 bytesQueued;
    struct LinuxQueue *queue = (struct LinuxQueue*) (handle);
    //PRINT_QUEUE(queue);
    if(checkQueue(queue) || !item) {
        pr_err("Cannot queue item, pointer is zero. handle=%p item=%p\n", queue, item);
        return -1;
    }
    bytesQueued = kfifo_in_spinlocked(&queue->fifo, item, itemSize, &queue->slock);
    wake_up(&queue->wq);
    return bytesQueued;
}

int fifo_len_spinlocked(struct kfifo_rec_ptr_2 *fifo, spinlock_t *slock)
{
    int avail = 0;
    unsigned long flags;
    spin_lock_irqsave(slock, flags);
    avail = kfifo_len(fifo);
    spin_unlock_irqrestore(slock, flags);
    return avail;
}

INT32 OsDequeueBytes(UINT32 handle, void* item, UINT32 itemSize)
{
    struct LinuxQueue *queue = (struct LinuxQueue*) (handle);
    //PRINT_QUEUE(queue);
    if(checkQueue(queue) || !item) {
        pr_err("Cannot dequeue item, pointer is zero. handle=%p item=%p\n", queue, item);
        msleep(1);
        return -1;
    }
    return kfifo_out_spinlocked(&queue->fifo, item, itemSize, &queue->slock);
}

INT32 OsDequeueBytesWait(UINT32 handle, void* item, UINT32 itemSize, UINT32 timeout_ms)
{
    INT32 bytesReceived = 0;
    struct LinuxQueue *queue = (struct LinuxQueue*) (handle);
    //PRINT_QUEUE(queue);
    if(checkQueue(queue) || !item) {
        pr_err("Cannot dequeue item, pointer is zero. handle=%p item=%p from %pS\n",
                queue, item, __builtin_return_address(0));
        msleep(1);
        return -1;
    }
    if(timeout_ms == OS_NO_WAIT) {
        bytesReceived = kfifo_out_spinlocked(&queue->fifo, item, itemSize, &queue->slock);
    }
    else if(timeout_ms == OS_WAIT_FOREVER) {
        wait_event(queue->wq, fifo_len_spinlocked(&queue->fifo, &queue->slock));
        bytesReceived = kfifo_out_spinlocked(&queue->fifo, item, itemSize, &queue->slock);
    }
    else {
        /* TODO: Note that this doesn't guarantee data if multiple threads are waiting on
         * this queue, since between wait_event_timeout returns and kfifo_out, someone
         * else could steal our data. */
        bytesReceived = wait_event_timeout(
                queue->wq, fifo_len_spinlocked(&queue->fifo, &queue->slock), timeout_ms * HZ/1000);
        if(bytesReceived) {
            bytesReceived = kfifo_out_spinlocked(&queue->fifo, item, itemSize, &queue->slock);
        }
    }
    return bytesReceived;
}

void OsQueueReset(UINT32 handle)
{
    struct LinuxQueue *queue = (struct LinuxQueue*) (handle);
    unsigned long flags;
    //PRINT_QUEUE(queue);
    if(checkQueue(queue)) {
        pr_err("Cannot delete pointer to zero queue, handle=%p\n", queue);
        return;
    }
    spin_lock_irqsave(&queue->slock, flags);
    kfifo_reset(&queue->fifo);
    spin_unlock_irqrestore(&queue->slock, flags);
}

UINT32 OsQueueSize(DM_RESOURCE_HANDLE handle)
{
    UINT32 bytesFree;
    unsigned long flags;
    struct LinuxQueue *queue = (struct LinuxQueue*) (handle);
    //PRINT_QUEUE(queue);
    if(checkQueue(queue)) {
        //BUG_ON(!queue);
        pr_err("Cannot dequeue item, pointer is zero. handle=%p\n", queue);
        return 0;
    }
    /* This only returns approximate size, since some objects in the queue
     * can be larger than others. */
    spin_lock_irqsave(&queue->slock, flags);
    bytesFree = kfifo_size(&queue->fifo);
    spin_unlock_irqrestore(&queue->slock, flags);
    return bytesFree;
}

UINT32 OsWaitEventGroupTimeout(DM_RESOURCE_HANDLE eventHandle, UINT32 mask, UINT32 waitTime)
{
    unsigned long flags;
    UINT32 wasSet = 0;
    struct LinuxEvent *le = (struct LinuxEvent*) (eventHandle);
    if(eventErrors(le, DM_LINUX_EVENT_TYPE_GROUP)) {
        pr_err("Cannot wait on null Event or wrong Event type.\n");
        return false;
    }
    if(waitTime == OS_NO_WAIT) {
        spin_lock_irqsave(&le->slock, flags);
        wasSet = le->events & mask;
        le->events &= ~wasSet;
        spin_unlock_irqrestore(&le->slock, flags);
    }
    else if(waitTime == OS_WAIT_FOREVER) {
        wait_event(le->wq, le->events & mask);
        spin_lock_irqsave(&le->slock, flags);
        wasSet = le->events & mask;
        le->events &= ~wasSet;
        spin_unlock_irqrestore(&le->slock, flags);
    }
    else {
        wasSet = wait_event_timeout(le->wq, le->events & mask, waitTime * HZ/1000);
        if(wasSet) {
            spin_lock_irqsave(&le->slock, flags);
            wasSet = le->events & mask;
            le->events &= ~wasSet;
            spin_unlock_irqrestore(&le->slock, flags);
        }
    }
    return wasSet;
}

/*
 * Waits for all the events in the mask to signal. Does NOT clear events.
 */
UINT32 OsWaitEventGroupAll(DM_RESOURCE_HANDLE eventHandle, UINT32 mask)
{
    unsigned long flags;
    UINT32 wasSet = 0;
    struct LinuxEvent *le = (struct LinuxEvent*) (eventHandle);
    if(eventErrors(le, DM_LINUX_EVENT_TYPE_GROUP)) {
        pr_err("Cannot wait on null Event or wrong Event type.\n");
        return false;
    }
    wait_event(le->wq, !((le->events & mask)^ mask));
    spin_lock_irqsave(&le->slock, flags);
    wasSet = le->events & mask;
    spin_unlock_irqrestore(&le->slock, flags);
    return wasSet;
}

void OsClearEvent(DM_RESOURCE_HANDLE eventHandle, UINT32 mask)
{
    unsigned long flags;
    struct LinuxEvent *le = (struct LinuxEvent*) (eventHandle);
    if(eventErrors(le, DM_LINUX_EVENT_TYPE_GROUP)) {
        pr_err("Cannot delete null event or wrong event type.\n");
        return;
    }
    spin_lock_irqsave(&le->slock, flags);
    le->events &= ~mask;
    spin_unlock_irqrestore(&le->slock, flags);
}

